<?php

namespace App\Http\Controllers;

use App\Models\Appointment;
use App\Models\Category;
use App\Models\Subscription;
use App\Models\User;
use Illuminate\Http\Request;

class AppointmentController extends Controller
{

    public function index()
    {
        $user = \Auth::user();
        if (\Auth::user()->can('manage appointment')) {
            if (\Auth::user()->type == 'employee') {
                $appointments = Appointment::where('user_id', '=', $user->id)->get();
            } else {
                $appointments = Appointment::where('parent_id', '=', $user->parentId())->get();
            }

        } else {
            return redirect()->back()->with('error', __('Permission Denied!'));
        }
        return view('appointment.index', compact('appointments'));
    }


    public function create()
    {
        $category = Category::where('parent_id', \Auth::user()->parentId())->get()->pluck('title', 'id');
        $category->prepend(__('Select Category'), '');

        $employees = User::where('parent_id', \Auth::user()->parentId())->whereIn('type', ['employee'])->get()->pluck('name', 'id');
        $employees->prepend(__('Select Employee'), '');

        $customers = User::where('parent_id', \Auth::user()->parentId())->whereIn('type', ['customer'])->get()->pluck('name', 'id');
        $customers->prepend(__('Select Customer'), '');

        $status = Appointment::$status;
        return view('appointment.create', compact('category', 'employees', 'customers', 'status'));
    }


    public function store(Request $request)
    {

        if (\Auth::user()->can('create appointment')) {
            $validator = \Validator::make(
                $request->all(), [
                    'user_id' => 'required',
                    'customer_id' => 'required',
                    'category' => 'required',
                    'service' => 'required',
                    'date' => 'required',
                    'start_time' => 'required',
                    'end_time' => 'required',
                    'status' => 'required',
                ]
            );
            if ($validator->fails()) {
                $messages = $validator->getMessageBag();

                return redirect()->back()->with('error', $messages->first());
            }

            $ids = \Auth::user()->parentId();
            $authUser = \App\Models\User::find($ids);
            $total_appointment = $authUser->totalAppointment();
            $subscription = Subscription::find($authUser->subscription);
            if ($total_appointment < $subscription->total_appointment || $subscription->total_appointment == 0) {
                $appointment = new Appointment();
                $appointment->appointment_id = $this->appointmentNumber();
                $appointment->user_id = $request->user_id;
                $appointment->customer_id = $request->customer_id;
                $appointment->category = $request->category;
                $appointment->service = $request->service;
                $appointment->date = $request->date;
                $appointment->start_time = $request->start_time;
                $appointment->end_time = $request->end_time;
                $appointment->status = $request->status;
                $appointment->payment_status = 'unpaid';
                $appointment->notes = $request->notes;
                $appointment->parent_id = \Auth::user()->parentId();
                $appointment->save();

                return redirect()->route('appointment.index')->with('success', __('Appointment successfully created!'));
            } else {
                return redirect()->back()->with('error', __('Your appointment limit is over, Please upgrade your subscription.'));
            }

        } else {
            return redirect()->back()->with('error', __('Permission Denied!'));
        }
    }


    public function show(Appointment $appointment)
    {
        if (\Auth::user()->can('show appointment')) {
            return view('appointment.show', compact('appointment'));
        } else {
            return redirect()->back()->with('error', __('Permission Denied!'));
        }

    }


    public function edit(Appointment $appointment)
    {
        $category = Category::where('parent_id', \Auth::user()->parentId())->get()->pluck('title', 'id');
        $category->prepend(__('Select Category'), '');

        $employees = User::where('parent_id', \Auth::user()->parentId())->whereIn('type', ['employee'])->get()->pluck('name', 'id');
        $employees->prepend(__('Select Employee'), '');

        $customers = User::where('parent_id', \Auth::user()->parentId())->whereIn('type', ['customer'])->get()->pluck('name', 'id');
        $customers->prepend(__('Select Customer'), '');

        $status = Appointment::$status;
        return view('appointment.edit', compact('category', 'employees', 'customers', 'status', 'appointment'));
    }


    public function update(Request $request, Appointment $appointment)
    {
        if (\Auth::user()->can('edit appointment')) {
            $validator = \Validator::make(
                $request->all(), [
                    'user_id' => 'required',
                    'customer_id' => 'required',
                    'category' => 'required',
                    'service' => 'required',
                    'date' => 'required',
                    'start_time' => 'required',
                    'end_time' => 'required',
                    'status' => 'required',
                ]
            );
            if ($validator->fails()) {
                $messages = $validator->getMessageBag();

                return redirect()->back()->with('error', $messages->first());
            }


            $appointment->user_id = $request->user_id;
            $appointment->customer_id = $request->customer_id;
            $appointment->category = $request->category;
            $appointment->service = $request->service;
            $appointment->date = $request->date;
            $appointment->start_time = $request->start_time;
            $appointment->end_time = $request->end_time;
            $appointment->status = $request->status;
            $appointment->notes = $request->notes;
            $appointment->save();

            return redirect()->route('appointment.index')->with('success', __('Appointment successfully updated!'));
        } else {
            return redirect()->back()->with('error', __('Permission Denied!'));
        }
    }


    public function destroy(Appointment $appointment)
    {
        if (\Auth::user()->can('delete appointment')) {

            $appointment->delete();

            return redirect()->route('appointment.index')->with('success', __('Appointment successfully deleted!'));
        } else {
            return redirect()->back()->with('error', __('Permission Denied!'));
        }
    }

    function appointmentNumber()
    {
        $latest = Appointment::where('parent_id', \Auth::user()->parentId())->latest()->first();
        if (!$latest) {
            return 1;
        }

        return $latest->appointment_id + 1;
    }

    public function paymentCreate($id)
    {
        $appointment = Appointment::find($id);
        $payment_method = Appointment::$payment_method;
        return view('payment.create', compact('payment_method', 'appointment'));
    }

    public function paymentStore(Request $request, $id)
    {

        if (\Auth::user()->can('create payment')) {
            $validator = \Validator::make(
                $request->all(), [
                    'payment_date' => 'required',
                    'amount' => 'required',
                    'payment_method' => 'required',
                ]
            );
            if ($validator->fails()) {
                $messages = $validator->getMessageBag();

                return redirect()->back()->with('error', $messages->first());
            }

            $appointment = Appointment::find($id);
            $appointment->payment_date = $request->payment_date;
            $appointment->amount = $request->amount;
            $appointment->payment_details = $request->payment_details;
            $appointment->payment_method = $request->payment_method;
            $appointment->payment_status = 'paid';
            $appointment->save();

            return redirect()->route('appointment.index')->with('success', __('Appointment successfully created!'));
        } else {
            return redirect()->back()->with('error', __('Permission Denied!'));
        }
    }

    public function calendar()
    {
        $user = \Auth::user();
        if (\Auth::user()->can('manage appointment')) {
            if (\Auth::user()->type == 'employee') {
                $appointments = Appointment::where('user_id', '=', $user->id)->get();
            } else {
                $appointments = Appointment::where('parent_id', '=', $user->parentId())->get();
            }

            $eventData =$currentMonth = [];
            foreach ($appointments as $appointment) {

                $customer = !empty($appointment->customers) ? $appointment->customers->name : '';
                $services = !empty($appointment->services) ? $appointment->services->title : '';
                $event = [
                    'title' => $customer . ' - ' . $services,
                    'start' => $appointment->date . 'T' . $appointment->start_time,
                    'end'   => $appointment->date . 'T' . $appointment->end_time,
                    'url'=>    route('appointment.show', $appointment->id),
                ];
                $eventData[] = $event;

                if(date('m',strtotime($appointment->date))== date('m')){
                    $currentMonthEvent = [
                        'title' => $customer . ' - ' . $services,
                        'date' => \Auth::user()->dateFormat($appointment->date),
                        'time'   => \Auth::user()->timeFormat($appointment->start_time) . '-' . \Auth::user()->timeFormat($appointment->end_time),
                        'url'=>    route('appointment.show', $appointment->id),
                    ];
                    $currentMonth[] = $currentMonthEvent;
                }
            }


        } else {
            return redirect()->back()->with('error', __('Permission Denied!'));
        }
        return view('appointment.calendar', compact('appointments','eventData','currentMonth'));
    }
}
